#!/usr/bin/env python3
"""
AI Chatbot Starter - A beginner's guide to creating chatbots in Python
This file progresses from simple to advanced chatbot concepts.
"""

import random
import re
from datetime import datetime

# ===== LEVEL 1: BASIC CHATBOT =====
print("="*50)
print("LEVEL 1: BASIC CHATBOT")
print("="*50)

def simple_chatbot():
    """A very basic chatbot with predefined responses"""
    
    print("Simple Bot: Hello! I'm a basic chatbot. Type 'quit' to exit.")
    
    # Simple responses dictionary
    responses = {
        "hello": ["Hi there!", "Hello!", "Hey! How are you?"],
        "how are you": ["I'm doing great!", "I'm fine, thanks!", "All good!"],
        "what's your name": ["I'm SimpleBot!", "My name is SimpleBot", "Call me SimpleBot"],
        "bye": ["Goodbye!", "See you later!", "Take care!"],
        "help": ["I can chat with you! Try saying hello, asking how I am, or say bye to leave."]
    }
    
    while True:
        user_input = input("You: ").lower().strip()
        
        if user_input == "quit":
            print("Simple Bot: Goodbye!")
            break
            
        # Check if user input matches any of our patterns
        response_found = False
        for pattern, replies in responses.items():
            if pattern in user_input:
                response = random.choice(replies)
                print(f"Simple Bot: {response}")
                response_found = True
                break
        
        if not response_found:
            print("Simple Bot: I don't understand that. Try saying 'help'.")

# ===== LEVEL 2: PATTERN-BASED CHATBOT =====
print("\n" + "="*50)
print("LEVEL 2: PATTERN-BASED CHATBOT")
print("="*50)

class PatternBot:
    """A more advanced chatbot using pattern matching"""
    
    def __init__(self):
        self.name = "PatternBot"
        self.patterns = [
            # Pattern: (regex_pattern, list_of_responses)
            (r"hello|hi|hey", ["Hello there!", "Hi! Nice to meet you!", "Hey! What's up?"]),
            (r"how are you|how do you feel", ["I'm doing well!", "Great, thanks for asking!", "I'm fantastic!"]),
            (r"what.*your name", ["I'm PatternBot!", "My name is PatternBot", "You can call me PatternBot"]),
            (r"what.*time", [f"It's {datetime.now().strftime('%H:%M')}", "Let me check... it's " + datetime.now().strftime('%I:%M %p')]),
            (r"what.*date|today", [f"Today is {datetime.now().strftime('%B %d, %Y')}"]),
            (r"tell me.*joke", ["Why don't scientists trust atoms? Because they make up everything!", 
                              "Why did the programmer quit his job? He didn't get arrays!", 
                              "What do you call a fake noodle? An impasta!"]),
            (r"bye|goodbye|see you", ["Goodbye!", "See you later!", "Take care!", "Until next time!"]),
            (r"help", ["I can chat about various topics! Try asking about time, date, jokes, or just say hello!"]),
        ]
    
    def get_response(self, user_input):
        """Find a response based on pattern matching"""
        user_input = user_input.lower().strip()
        
        for pattern, responses in self.patterns:
            if re.search(pattern, user_input):
                return random.choice(responses)
        
        # Default responses if no pattern matches
        default_responses = [
            "That's interesting! Tell me more.",
            "I'm not sure about that. Can you rephrase?",
            "Hmm, I don't quite understand. Try asking something else!",
            "That's a good point! What else would you like to know?"
        ]
        return random.choice(default_responses)
    
    def chat(self):
        """Main chat loop"""
        print(f"{self.name}: Hello! I'm an improved chatbot. Type 'quit' to exit.")
        
        while True:
            user_input = input("You: ").strip()
            
            if user_input.lower() == "quit":
                print(f"{self.name}: Thanks for chatting! Goodbye!")
                break
            
            if user_input:  # Only respond if user typed something
                response = self.get_response(user_input)
                print(f"{self.name}: {response}")

# ===== LEVEL 3: CONTEXT-AWARE CHATBOT =====
print("\n" + "="*50)
print("LEVEL 3: CONTEXT-AWARE CHATBOT")
print("="*50)

class SmartBot:
    """A chatbot that remembers context and user information"""
    
    def __init__(self):
        self.name = "SmartBot"
        self.user_name = None
        self.conversation_history = []
        self.user_preferences = {}
        
    def remember_user_info(self, user_input):
        """Extract and remember user information"""
        # Try to extract user's name
        name_patterns = [
            r"my name is (\w+)",
            r"i'm (\w+)",
            r"call me (\w+)"
        ]
        
        for pattern in name_patterns:
            match = re.search(pattern, user_input.lower())
            if match:
                self.user_name = match.group(1).capitalize()
                return f"Nice to meet you, {self.user_name}!"
        
        return None
    
    def get_contextual_response(self, user_input):
        """Generate response based on context and history"""
        user_input_lower = user_input.lower().strip()
        
        # Store conversation
        self.conversation_history.append(user_input)
        
        # Check for name introduction
        name_response = self.remember_user_info(user_input)
        if name_response:
            return name_response
        
        # Personalized responses if we know the user's name
        if self.user_name:
            personal_patterns = [
                (r"hello|hi|hey", [f"Hello {self.user_name}!", f"Hi there, {self.user_name}!"]),
                (r"how are you", [f"I'm doing well, {self.user_name}! How about you?"]),
            ]
            
            for pattern, responses in personal_patterns:
                if re.search(pattern, user_input_lower):
                    return random.choice(responses)
        
        # Context-aware responses
        if len(self.conversation_history) > 1:
            last_message = self.conversation_history[-2].lower()
            
            if "joke" in last_message and "another" in user_input_lower:
                jokes = [
                    "Why don't programmers like nature? It has too many bugs!",
                    "How many programmers does it take to change a light bulb? None, that's a hardware problem!",
                    "Why do Java developers wear glasses? Because they don't C#!"
                ]
                return random.choice(jokes)
        
        # Default patterns (same as PatternBot but enhanced)
        patterns = [
            (r"hello|hi|hey", ["Hello!", "Hi there!", "Hey! Good to see you!"]),
            (r"how are you", ["I'm doing great!", "Fantastic, thanks for asking!"]),
            (r"what.*your name", ["I'm SmartBot, your intelligent assistant!"]),
            (r"tell me.*joke", ["Here's one: Why did the chatbot go to therapy? It had too many issues to process!"]),
            (r"bye|goodbye", ["Goodbye! It was great chatting with you!"]),
        ]
        
        for pattern, responses in patterns:
            if re.search(pattern, user_input_lower):
                return random.choice(responses)
        
        return "That's interesting! What else would you like to talk about?"
    
    def chat(self):
        """Main chat loop with context awareness"""
        print(f"{self.name}: Hello! I'm a smart chatbot that learns about you. What's your name?")
        
        while True:
            user_input = input("You: ").strip()
            
            if user_input.lower() == "quit":
                farewell = f"Goodbye, {self.user_name}!" if self.user_name else "Goodbye!"
                print(f"{self.name}: {farewell} Thanks for the great conversation!")
                break
            
            if user_input:
                response = self.get_contextual_response(user_input)
                print(f"{self.name}: {response}")

# ===== LEVEL 4: AI-POWERED CHATBOT SETUP =====
print("\n" + "="*50)
print("LEVEL 4: AI-POWERED CHATBOT (Setup Guide)")
print("="*50)

def ai_chatbot_guide():
    """Guide for creating AI-powered chatbots"""
    
    guide = """
    🤖 AI-POWERED CHATBOT SETUP GUIDE
    
    To create a truly intelligent chatbot, you'll want to use AI APIs:
    
    1. OpenAI API (ChatGPT):
       - Sign up at: https://platform.openai.com
       - Install: pip install openai
       - Use GPT-3.5 or GPT-4 for responses
    
    2. Google Gemini API:
       - Sign up at: https://ai.google.dev
       - Install: pip install google-generativeai
       - Use Gemini for conversational AI
    
    3. Hugging Face Transformers:
       - Install: pip install transformers torch
       - Use pre-trained models locally
       - Models like DialoGPT, BlenderBot
    
    4. Free Alternatives:
       - Ollama (run models locally)
       - Cohere API (has free tier)
       - Anthropic Claude API
    
    Example with OpenAI (you'll need an API key):
    
    ```python
    import openai
    
    openai.api_key = "your-api-key-here"
    
    def ai_response(user_message):
        response = openai.ChatCompletion.create(
            model="gpt-3.5-turbo",
            messages=[{"role": "user", "content": user_message}]
        )
        return response.choices[0].message.content
    ```
    
    🚀 NEXT STEPS FOR LEARNING:
    1. Start with the basic chatbot above
    2. Learn about APIs and HTTP requests
    3. Practice with pattern matching and regex
    4. Explore natural language processing (NLP)
    5. Learn about machine learning basics
    6. Try integrating with AI services
    
    📚 RECOMMENDED LEARNING PATH:
    - Week 1-2: Master basic Python and the examples above
    - Week 3-4: Learn about APIs and JSON
    - Week 5-6: Try integrating with a free AI service
    - Week 7-8: Add features like memory, personality, etc.
    """
    
    print(guide)

# ===== MAIN DEMO FUNCTION =====
def run_chatbot_demo():
    """Run different chatbot levels for demonstration"""
    
    while True:
        print("\n" + "="*60)
        print("🤖 CHATBOT LEARNING DEMO")
        print("="*60)
        print("Choose which chatbot to try:")
        print("1. Simple Chatbot (Basic responses)")
        print("2. Pattern Bot (Regex matching)")
        print("3. Smart Bot (Context-aware)")
        print("4. AI Setup Guide")
        print("5. Exit")
        
        choice = input("\nEnter your choice (1-5): ").strip()
        
        if choice == "1":
            simple_chatbot()
        elif choice == "2":
            bot = PatternBot()
            bot.chat()
        elif choice == "3":
            bot = SmartBot()
            bot.chat()
        elif choice == "4":
            ai_chatbot_guide()
        elif choice == "5":
            print("Thanks for exploring chatbots! Happy coding! 🚀")
            break
        else:
            print("Invalid choice. Please try again.")

# ===== EXERCISES FOR PRACTICE =====
print("\n" + "="*50)
print("🎯 PRACTICE EXERCISES")
print("="*50)

exercises = """
Try these exercises to improve your chatbot skills:

1. BEGINNER:
   - Add more response patterns to the simple chatbot
   - Make the bot remember the user's favorite color
   - Add a feature to tell random facts

2. INTERMEDIATE:
   - Create a chatbot that can do simple math
   - Add a mood system (happy, sad, excited)
   - Make the bot ask questions back to the user

3. ADVANCED:
   - Save conversation history to a file
   - Create different personalities for the bot
   - Add a learning system where the bot improves over time

4. EXPERT:
   - Integrate with a weather API
   - Add voice recognition (speech-to-text)
   - Create a web interface using Flask
"""

print(exercises)

# To run the demo, uncomment the line below:
# run_chatbot_demo()

print("\n🎉 Ready to start building chatbots!")
print("Run this file and uncomment the last line to try the interactive demo!")
